# Google 2FA App Password Generator Installer for Windows
# Author: @systemaudit
# Enhanced with architecture detection and fallback methods

Write-Host ""
Write-Host "==========================================" -ForegroundColor Green
Write-Host " Google 2FA App Password Generator" -ForegroundColor Green
Write-Host " Installer for Windows (PowerShell)" -ForegroundColor Green  
Write-Host "==========================================" -ForegroundColor Green
Write-Host ""

# Detect Architecture
$arch = [System.Environment]::GetEnvironmentVariable("PROCESSOR_ARCHITECTURE")
switch ($arch) {
    "AMD64" { $archName = "x64" }
    "ARM64" { $archName = "ARM64" }
    "x86"   { $archName = "x86" }
    default { $archName = "Unknown" }
}
Write-Host "Architecture: $archName" -ForegroundColor Blue
Write-Host ""

# Function to check Node.js version
function Test-NodeVersion {
    try {
        $version = node -v
        $major = [int]($version -replace 'v(\d+)\..*', '$1')
        return $major -ge 16
    } catch {
        return $false
    }
}

# Function to install Node.js via Chocolatey
function Install-NodeChocolatey {
    Write-Host "Attempting to install Node.js via Chocolatey..." -ForegroundColor Yellow
    
    if (Get-Command choco -ErrorAction SilentlyContinue) {
        choco install nodejs-lts -y
        if ($LASTEXITCODE -eq 0) {
            refreshenv
            Write-Host "✓ Node.js installed via Chocolatey" -ForegroundColor Green
            return $true
        }
    }
    return $false
}

# Function to install Node.js via Scoop
function Install-NodeScoop {
    Write-Host "Attempting to install Node.js via Scoop..." -ForegroundColor Yellow
    
    if (Get-Command scoop -ErrorAction SilentlyContinue) {
        scoop install nodejs-lts
        if ($LASTEXITCODE -eq 0) {
            Write-Host "✓ Node.js installed via Scoop" -ForegroundColor Green
            return $true
        }
    }
    return $false
}

# Function to install Node.js via winget
function Install-NodeWinget {
    Write-Host "Attempting to install Node.js via winget..." -ForegroundColor Yellow
    
    if (Get-Command winget -ErrorAction SilentlyContinue) {
        winget install OpenJS.NodeJS.LTS --silent --accept-package-agreements --accept-source-agreements
        if ($LASTEXITCODE -eq 0) {
            # Refresh PATH
            $env:Path = [System.Environment]::GetEnvironmentVariable("Path","Machine") + ";" + [System.Environment]::GetEnvironmentVariable("Path","User")
            Write-Host "✓ Node.js installed via winget" -ForegroundColor Green
            return $true
        }
    }
    return $false
}

# Check Node.js
Write-Host "Checking Node.js..." -ForegroundColor Yellow
$nodeInstalled = $false

if (Get-Command node -ErrorAction SilentlyContinue) {
    if (Test-NodeVersion) {
        $nodeVersion = node -v
        Write-Host "✓ Node.js $nodeVersion" -ForegroundColor Green
        $nodeInstalled = $true
    } else {
        Write-Host "Node.js version is too old. Minimum required: v16.0.0" -ForegroundColor Yellow
    }
}

# Install Node.js if needed
if (-not $nodeInstalled) {
    Write-Host "Installing Node.js..." -ForegroundColor Yellow
    Write-Host ""
    
    # Try multiple methods
    if (Install-NodeChocolatey) {
        $nodeInstalled = $true
    } elseif (Install-NodeScoop) {
        $nodeInstalled = $true
    } elseif (Install-NodeWinget) {
        $nodeInstalled = $true
    } else {
        Write-Host "Automatic installation failed!" -ForegroundColor Red
        Write-Host ""
        Write-Host "Please install Node.js manually:" -ForegroundColor Yellow
        Write-Host "  1. Download from: https://nodejs.org" -ForegroundColor White
        Write-Host "  2. Choose LTS version (18.x or higher)" -ForegroundColor White
        Write-Host "  3. Run the installer" -ForegroundColor White
        Write-Host "  4. Restart this script after installation" -ForegroundColor White
        Write-Host ""
        Read-Host "Press Enter to exit"
        exit 1
    }
}

# Check npm
try {
    $npmVersion = npm -v
    Write-Host "✓ npm $npmVersion" -ForegroundColor Green
} catch {
    Write-Host "Error: npm is not installed!" -ForegroundColor Red
    Write-Host "Please reinstall Node.js from: https://nodejs.org" -ForegroundColor Yellow
    Read-Host "Press Enter to exit"
    exit 1
}

Write-Host ""

# Set install directory
$installDir = "$env:USERPROFILE\google-2fa-apppassword"

# Download or update
if (Test-Path $installDir) {
    Write-Host "Directory exists. Updating..." -ForegroundColor Yellow
    
    # Backup .env if exists
    if (Test-Path "$installDir\.env") {
        Write-Host "Backing up .env file..." -ForegroundColor Blue
        Copy-Item "$installDir\.env" "$installDir\.env.backup"
    }
    
    # Download latest version
    Write-Host "Downloading latest version..." -ForegroundColor Yellow
    $zipPath = "$env:TEMP\appassword.zip"
    Invoke-WebRequest -Uri "https://mirror.malingjenius.ru/project/appassword/latest.zip" -OutFile $zipPath
    
    # Remove old files (except .env.backup)
    Get-ChildItem -Path $installDir -Exclude ".env.backup" | Remove-Item -Recurse -Force
    
    # Extract new version
    Write-Host "Extracting files..." -ForegroundColor Yellow
    Expand-Archive -Path $zipPath -DestinationPath $installDir -Force
    Remove-Item $zipPath
    
    # Restore .env
    if (Test-Path "$installDir\.env.backup") {
        Move-Item "$installDir\.env.backup" "$installDir\.env" -Force
    }
    
    Write-Host "[OK] Updated to latest version" -ForegroundColor Green
} else {
    Write-Host "Downloading and installing..." -ForegroundColor Yellow
    
    # Download latest version
    $zipPath = "$env:TEMP\appassword.zip"
    Invoke-WebRequest -Uri "https://mirror.malingjenius.ru/project/appassword/latest.zip" -OutFile $zipPath
    
    # Create directory
    New-Item -ItemType Directory -Path $installDir | Out-Null
    
    # Extract
    Write-Host "Extracting files..." -ForegroundColor Yellow
    Expand-Archive -Path $zipPath -DestinationPath $installDir
    Remove-Item $zipPath
    
    Write-Host "[OK] Installation complete" -ForegroundColor Green
}

Set-Location $installDir

# Install dependencies
Write-Host ""
Write-Host "Installing dependencies..." -ForegroundColor Yellow
npm install | Out-Null
Write-Host "[OK] Dependencies installed" -ForegroundColor Green

# Install Playwright
Write-Host ""
Write-Host "Installing Playwright..." -ForegroundColor Yellow
npx playwright install chromium
Write-Host "[OK] Playwright installed" -ForegroundColor Green

# Create directories
Write-Host ""
Write-Host "Creating directories..." -ForegroundColor Yellow
New-Item -ItemType Directory -Force -Path logs | Out-Null
New-Item -ItemType Directory -Force -Path output | Out-Null
Write-Host "[OK] Directories created" -ForegroundColor Green

# Create .env if not exists
if (-not (Test-Path ".env")) {
    Write-Host ""
    Write-Host "Creating .env file..." -ForegroundColor Yellow
    @"
# Browser Settings
HEADLESS=true
DEBUG_MODE=false

# Timeouts (ms) - Balanced for reliability
NAVIGATION_TIMEOUT=18000
IMPLICIT_TIMEOUT=7000
VERIFICATION_TIMEOUT=10000

# Processing - Balanced
BATCH_SIZE=3
BATCH_DELAY=4000

# TOTP API
TOTP_API_URL=https://2fa.live/tok

# Output
OUTPUT_FORMAT=csv
OUTPUT_PATH=output/results.csv
"@ | Out-File -FilePath ".env" -Encoding UTF8
    Write-Host "[OK] .env file created" -ForegroundColor Green
}

# Create desktop shortcut
Write-Host ""
Write-Host "Creating desktop shortcut..." -ForegroundColor Yellow
$desktop = [Environment]::GetFolderPath("Desktop")
$shell = New-Object -ComObject WScript.Shell
$shortcut = $shell.CreateShortcut("$desktop\Google 2FA Generator.lnk")
$shortcut.TargetPath = "cmd.exe"
$shortcut.Arguments = "/k cd /d `"$installDir`" && npm start"
$shortcut.WorkingDirectory = $installDir
$shortcut.Save()
Write-Host "[OK] Desktop shortcut created" -ForegroundColor Green

Write-Host ""
Write-Host "==========================================" -ForegroundColor Green
Write-Host " Installation Complete!" -ForegroundColor Green
Write-Host "==========================================" -ForegroundColor Green
Write-Host ""
Write-Host "Location: $installDir" -ForegroundColor Cyan
Write-Host "Architecture: $archName" -ForegroundColor Cyan
Write-Host ""
Write-Host "Next steps:" -ForegroundColor Yellow
Write-Host "1. Edit .env (optional): notepad `"$installDir\.env`"" -ForegroundColor White
Write-Host "2. Run: Desktop shortcut or 'npm start'" -ForegroundColor White
Write-Host ""
Read-Host "Press Enter to exit"
